/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY.                         *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.  Contact information: bergmark@cs.cornell.edu     *
 ******************************************************************************/
package cnrg.itx.signal;

import cnrg.itx.ds.*;
import cnrg.itx.datax.*;
import cnrg.itx.signal.SignalID;

import java.io.*;		   

/**
 * This class extends SigPacket and is used in the 3-way handshake during a Dial Sequence.
 */
public class InvitePacket extends SigPacket implements Serializable
{
	/** Source userID **/
	protected UserID source;
	/** Source location **/
	protected Location srcLoc;
	/** Source application's string description **/
	protected String appDesc = "";
	
	/** Result of query to user **/
	protected int resultFlag = SignalID.INVALID;
	/** Source's data exchange properties **/
	protected PropertiesCollection myPC = null;
	/** Reason for rejecting a call **/
	protected String reason = "";
	protected String myCallID = null;
	/** Sequence number for connection belonging to the recipient DesktopSignaling **/
	protected Long myConnSeqNumber = null;
	/** Sequence number belonging to the initiating DesktopSignaling **/
	protected Long myCSeq = null;
	
	/**
	 * Default Constructor for the InvitePacket
	 * 
	 * @param   None
	 * @return  void
	 * 
	 */
	public InvitePacket(){
		this(null, null, "", SignalID.INVALID);
	}
	
	/**
	 * Overloaded Constructor for the InvitePacket.
	 * 
	 * @param   s is the UserID object representing the source
	 * @param   sl is the Location object representing the source
	 * @param   ad is the String description of the Source
	 * @param   meth is the integer representing the method id
	 * 
	 * @return  void
	 * 
	 */
	public InvitePacket(UserID s, Location sl, String ad, int meth){
		super(meth,SignalID.INVITE);
		source = s;
		srcLoc = sl;		
		appDesc = ad;
	}
	
	/**
	 * Returns the UserID associated with the sender of the InvitePacket
	 * 
	 * @param   None
	 * @return  UserID object of the sender
	 */
	public UserID getSenderID(){
		return source;
	}

	/**
	 * Returns the Location associated with the sender of the InvitePacket
	 * 
	 * @param   None
	 * @return  Location object of the sender
	 */
	public Location getSenderLoc(){
		return srcLoc;
	}
	
	/**
	 * Returns the description of the sender of the InvitePacket
	 * 
	 * @param   None
	 * @return  a String description of the sender
	 */
	public String getDescription(){
		return appDesc;
	}

	/**
	 * Returns the reason an invitation was rejected
	 * 
	 * @param   None
	 * @return  a String describing the reason for rejecting the invitation
	 */
	public String getReason(){
		return reason;
	}
	
	/**
	 * Returns the senders' PropertiesCollection
	 * 
	 * @param   None
	 * @return  senders' PropertiesCollection object
	 */	
	public PropertiesCollection getPropertiesCollection(){
		return myPC;
	}
	
	/**
	 * Returns the call-id of the current call session
	 * 
	 * @return the call-id
	 */
	public String getCallID()
	{
		return myCallID;
	}
	
	/**
	 * Returns the initiating DesktopSignaling's connection sequence number
	 * 
	 * @return the sequence number
	 */
	public Long getCSeq()
	{
		return myCSeq;
	}

	/**
	 * Assigns the source UserID
	 * 
	 * @param   s is the UserID object representing the sender
	 * @return  void
	 */	
	public void setSourceID(UserID s){
		source = s;
	}

	/**
	 * Assigns the source Location
	 * 
	 * @param   l is the Location object representing the sender's location
	 * @return  void
	 */		
	public void setSourceLoc(Location l){
		srcLoc = l;
	}

	/**
	 * Sets the source description
	 * 
	 * @param   s is the String representing the sender's description
	 * @return  void
	 */		
	public void setAppDescription(String s){
		appDesc = s;
	}

	/**
	 * Assigns the source's PropertiesCollection for Data Exchange
	 * 
	 * @param   pc is the PropertiesCollection object representing the sender's PropertiesCollection
	 * @return  void
	 */		
	public void setPropertiesCollection(PropertiesCollection pc){
		myPC = pc;
	}

	/**
	 * Assigns the sequence number of the recipient for the packet. A sequence number is exchanged in the 3-way handshake
	 * 
	 * @param   s is the Long object representing the sequence number
	 * @return  void
	 */		
	public void setSeqNumber(Long s){
		myConnSeqNumber = s;
	}
	
	/**
	 * Returns the sequence number associated with the recipient's InvitePacket
	 * 
	 * @param   None
	 * @return  the Long object representing the sequence number
	 */
	public Long getSeqNumber(){
		return myConnSeqNumber;
	}
	
	/**
	 * Assigns the call-id of the current session.
	 * 
	 * @param s the call-id
	 */
	public void setCallID(String s)
	{
		myCallID = s;
	}
	
	/**
	 * Assigns the sequence number belonging to the initiating DesktopSignaling.
	 * 
	 * @param s the sequence number
	 */
	public void setCSeq(Long s)
	{
		myCSeq = s;
	}
	
	/**
	 * Applications call this method to indicate that they accept the invitation.
	 * 
	 * @param   None
	 * @return  void
	 * 
	 */	
	public void accept(){
		resultFlag = SignalID.ACCEPT;
	}

	/**
	 * Applications call this method to indicate that they reject the invitation.
	 * 
	 * @param   r is the reason for rejection
	 * @return  void
	 * 
	 */		
	public void reject(String r){
		resultFlag = SignalID.REJECT;
		reason = r;
	}
	
	/**
	 * Applications call this method to indicate that they are busy at present and cannot accept 
	 * the invitation.
	 * 
	 * @param   None
	 * @return  void
	 * 
	 */		
	public void busy(){
		resultFlag = SignalID.BUSY;
	}
	
	/**
	 * Applications call this method to confirm a sequence.
	 * 
	 * @param   None
	 * @return  void
	 * 
	 */
	public void confirm(){
		resultFlag = SignalID.CONFIRMED;
	}
	
	/**
	 * Checks whether the Invitation sent was accepted by the peer application.
	 * 
	 * @param   None.
	 * @return	TRUE if invitation was accepted, FALSE otherwise 
	 * 
	 */
	public boolean wasAccepted(){
		return (resultFlag == SignalID.ACCEPT);
	}


	/**
	 * Checks whether the Invitation was rejected by the peer application.
	 * 
	 * @param   None.
	 * @return	TRUE if invitation was rejected, FALSE otherwise 
	 * 
	 */
	public boolean wasRejected(){
		return (resultFlag == SignalID.REJECT);
	}	

	/**
	 * Checks whether the peer application was busy.
	 * 
	 * @param   None.
	 * @return	TRUE if peer was busy, FALSE otherwise 
	 * 
	 */
	public boolean wasBusy(){
		return (resultFlag == SignalID.BUSY);
	}
	
	/**
	 * Checks whether it is a confirm packet .
	 * 
	 * @param   None.
	 * @return	TRUE if it is a confirm packet, FALSE otherwise 
	 * 
	 */
	public boolean wasConfirmed(){
		return (resultFlag == SignalID.CONFIRMED);
	}
	
	
}
